% Copyright (c) 2016 Giampaolo D'Alessandro
% Mathematical Sciences - University of Southampton
% Email: dales@soton.ac.uk
% 
% Permission is hereby granted, free of charge, to any person obtaining a copy
% of this software and associated documentation files (the "Software"), to deal
% in the Software without restriction, including without limitation the rights
% to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
% copies of the Software, and to permit persons to whom the Software is
% furnished to do so, subject to the following conditions:
% 
% The above copyright notice and this permission notice shall be included in
% all copies or substantial portions of the Software.
% 
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
% IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
% AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
% OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
% SOFTWARE.

function [x,y,T] = ...
  ThermalProfile(L_G,Lx,Ly,nx,ny,z,alpha,kappa,sigma,q,W)
% ThermalProfile - Temperature profile inside a liquid crystal cell. 
%
% This function solve the non-dimensional heat diffusion equation in the
% presence of a Gaussian heating beam.  x is the coordinate along the
% grating pattern, y is the other coordinate in the plane of the cell and z
% is the coordinate into the cell.
%
% Input parameters
% L_G - The thickness of the glass layers
% Lx, Ly - The size of the domain in the x and y directions.
% nx,ny - The number of grid points in the x and y directions.
% z (vector of size nz) - The z coordinates
% alpha - Function handle that computes the pump absorption in the LC as a
% function of z.
% kappa - The heat conductivity of the glass relative to that of the LC
% sigma - The 1/e width of the Gaussian pump intensity beam.
% q - The wave number of the grating
% W - Weights for the definite Clencurt quadrature.
%
% Output parameters
% x,y,z - Vectors that contain the coordinates along the corresponding
% axes.
% T (array [nx,ny,nz]- The temperature profile in the cell.

% Create the grids in coordinate space: we use a periodic grid in the x and
% y directions and a Gauss-Lobatto grid in the z-direction.
x = linspace(-0.5*Lx,0.5*Lx,nx+1);
x = x(1:nx);
y = linspace(-0.5*Ly,0.5*Ly,ny+1);
y = y(1:ny);
[X,Y] = meshgrid(x,y);
nz = length(z);
[D1,~] = cheb(nz-1);

% Create the corresponding grids in Fourier space
if (nx>1)
  kx_max = 0.5*2*pi/(x(2)-x(1));
  k_x = linspace(-kx_max,kx_max,nx+1);
  k_x = fftshift(k_x(1:nx));
else
  k_x = 0;
end
if (ny > 0)
  ky_max = 0.5*2*pi/(y(2)-y(1));
  k_y = linspace(-ky_max,ky_max,ny+1);
  k_y = fftshift(k_y(1:ny));
else
  k_y = 0;
end
[KX,KY] = meshgrid(k_x,k_y);
k2 = KX.^2 + KY.^2;
k = sqrt(k2);

% Compute the pump profile in coordinate space...
Pump = exp(-(X.^2+Y.^2)/sigma^2).*(1+cos(q*X));
% ... and in Fourier space
Pump_F = fft2(Pump);
% Compute the pump attenuation factor during propagation in the slice
P_abs = exp(-W*alpha(z));

% Solve the heat diffusion equation numerically in Fourier space

% We first need to define the second derivative matrix in the z direction
D = D1*D1;
% Define the output variable
T = zeros([ny,nx,nz]);
% Solve the heat diffusion equation for each point in the Fourier plane
parfor m = 1:nx
  for n = 1:ny
    % Impose the boundary conditions:
    % %
    % \begin{equation*}
    %    T'(\pm d) \pm \frac{\kappa k}{\tanh(k L_{G})} = 0 \, ,
    % \end{equation*}
    % %
    % where $k^2 = k_{x}^2 + k_{y}^2$ and $\kappa = k_{G}/k_{LC}$, with
    % $k_{G}$ and $k_{LC}$ the thermal conductivity of glass and liquid
    % crystal respectively.
    D_op = D - k2(n,m)*eye(nz);
    if (k(n,m) ~= 0)
      D_op(1,:) = D1(1,:);
      D_op(1,1) = D_op(1,1) + kappa*k(n,m)/tanh(k(n,m)*L_G);
      D_op(end,:) = D1(end,:);
      D_op(end,end) = D_op(end,end) - kappa*k(n,m)/tanh(k(n,m)*L_G);
    else
      D_op(1,:) = D1(1,:);
      D_op(1,1) = D_op(1,1) + kappa/L_G;
      D_op(end,:) = D1(end,:);
      D_op(end,end) = D_op(end,end) - kappa/L_G;
    end
    % Build the right hand side vector 
    f = -alpha(z).*Pump_F(n,m).*P_abs;
    f(1) = 0; f(end) = 0;
    % Solve for the temperature profile
    T(n,m,:) = D_op\f;
  end
end

% Write the temperature in coordinate space

for j = 1:nz
  T(:,:,j) = real(ifft2(squeeze(T(:,:,j))));
end

end